// Copyright (c) 2016 Noriaki Mitsunaga
// You can use under GPL version 2 or later

#include <TimeLib.h> // Time Library https://github.com/PaulStoffregen/Time
#include <ESP8266WiFi.h>
#include <WiFiUdp.h>
extern "C" {
#include "user_interface.h"
}

struct Setting {
  char *ssid;
  char *pass;
  char *ntpServerName;
};

struct Setting settings[] = {{
    "SSID1",
    "PASS1",
    "NTPserver1"
  },
  { "SSID2",
    "PASS2",
    "NTPserver2"
  }
};
const int numOfSettings = sizeof(settings)/sizeof(settings[0]);

struct Setting *setting;

unsigned int localPort = 2390;      // local port to listen for UDP packets

/* Don't hardwire the IP address or we won't get the benefits of the pool.
    Lookup the IP address for the host name instead */
//IPAddress timeServer(129, 6, 15, 28); // time.nist.gov NTP server
IPAddress timeServerIP; // time.nist.gov NTP server address

const int NTP_PACKET_SIZE = 48; // NTP time stamp is in the first 48 bytes of the message

byte packetBuffer[ NTP_PACKET_SIZE]; //buffer to hold incoming and outgoing packets

// A UDP instance to let us send and receive packets over UDP
WiFiUDP udp;

const int LEDpin = 5;

void setup()
{
  pinMode(LEDpin, OUTPUT);

  Serial.begin(115200);
  Serial.println();
  Serial.println();

  // We start by connecting to a WiFi network
  for (int id = 0;;) {
    setting = &settings[id];
    Serial.print("Connecting to ");
    Serial.println(setting->ssid);
    WiFi.begin(setting->ssid, setting->pass);

    int s;
    for (int i = 0; i < 40; i ++) {
      s = WiFi.status();
      if (s == WL_CONNECTED ||
          s == WL_NO_SSID_AVAIL ||
          s == WL_CONNECT_FAILED)
        break;
      delay(500);
      Serial.print(".");
    }
    Serial.println("");
    if (s == WL_CONNECTED)
      break;
    WiFi.disconnect();
    delay(1000);
    id ++;
    if (id>= numOfSettings)
      id = 0;
  }

  Serial.println("WiFi connected");
  Serial.println("IP address: ");
  Serial.println(WiFi.localIP());

  Serial.println("Starting UDP");
  udp.begin(localPort);
  Serial.print("Local port: ");
  Serial.println(udp.localPort());
  setTimeByNTP();
  wifi_set_sleep_type(LIGHT_SLEEP_T);
}

uint32_t prev = 0;
time_t t = 0;

void loop()
{
  if (t == now()) {
    delay(50);
    return;
  }

  t = now();
  tmElements_t tm;
  breakTime(t, tm);

  char s[20];

  if ((tm.Second % 60) == 59 && millis() - prev > 10 * 1000) {
    digitalWrite(LEDpin, HIGH);
    setTimeByNTP();

    t = now();
    tmElements_t tm;
    breakTime(t, tm);
    const char* format = "%04d%02d%02d%02d%02d%02d";
    sprintf(s, format, tm.Year + 1970, tm.Month, tm.Day, tm.Hour, tm.Minute, tm.Second);
    Serial.println(s);
    digitalWrite(LEDpin, LOW);
    prev = millis();
  } else {
    delay(900);
  }
}

void setTimeByNTP()
{
  //get a random server from the pool
  WiFi.hostByName(setting->ntpServerName, timeServerIP);

  sendNTPpacket(timeServerIP); // send an NTP packet to a time server
  unsigned long sendMillis = millis();
  unsigned long recvMillis = 0;

  int cb;
  for (int i = 0; i < 20; i++) {
    cb = udp.parsePacket();
    if (cb > 0) {
      recvMillis = millis();
      break;
    }
    //ESP.wdtFeed();
    delay(50);
  }

  if (!cb) {
    Serial.println("no packet yet");
  }
  else {
    // We've received a packet, read the data from it
    uint32_t ms;
    time_t epoch = recvNTPPacket(ms);
    unsigned long delta = (recvMillis - sendMillis) / 2 + ms;
    epoch += (delta / 1000);
    delta = delta % 1000;
    time_t dt = delta;
    //ESP.wdtFeed(); // XXX
    //Serial.println(dt);
    if (dt > 300) {
      delay(dt);
      setTime(epoch + 3600 * 9 + 1);
    } else {
      setTime(epoch + 3600 * 9);
    }
    //ESP.wdtFeed(); // XXX
    //uint32_t prev = setTimeOffset(dt);
    //Serial.println(prev);
    //setPrevMillis(prev);
  }
}

// send an NTP request to the time server at the given address
unsigned long sendNTPpacket(IPAddress& address)
{
  Serial.println("sending NTP packet...");
  // set all bytes in the buffer to 0
  memset(packetBuffer, 0, NTP_PACKET_SIZE);
  // Initialize values needed to form NTP request
  // (see URL above for details on the packets)
  packetBuffer[0] = 0b11100011;   // LI, Version, Mode
  packetBuffer[1] = 0;     // Stratum, or type of clock
  packetBuffer[2] = 6;     // Polling Interval
  packetBuffer[3] = 0xEC;  // Peer Clock Precision
  // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12]  = 49;
  packetBuffer[13]  = 0x4E;
  packetBuffer[14]  = 49;
  packetBuffer[15]  = 52;

  // all NTP fields have been given values, now
  // you can send a packet requesting a timestamp:
  udp.beginPacket(address, 123); //NTP requests are to port 123
  udp.write(packetBuffer, NTP_PACKET_SIZE);
  udp.endPacket();
}

uint32_t recvNTPPacket(uint32_t &ms)
{
  udp.read(packetBuffer, NTP_PACKET_SIZE); // read the packet into the buffer

  //the timestamp starts at byte 40 of the received packet and is four bytes,
  // or two words, long. First, esxtract the two words:

  unsigned long highWord = word(packetBuffer[40], packetBuffer[41]);
  unsigned long lowWord = word(packetBuffer[42], packetBuffer[43]);
  // combine the four bytes (two words) into a long integer
  // this is NTP time (seconds since Jan 1 1900):
  uint32_t secsSince1900 = highWord << 16 | lowWord;
  //Serial.print("Seconds since Jan 1 1900 = " );
  //Serial.println(secsSince1900);

  highWord = word(packetBuffer[44], packetBuffer[45]);
  lowWord = word(packetBuffer[46], packetBuffer[47]);

  ms = highWord * 1000 / 65536;
  //Serial.println(s);

  // now convert NTP time into everyday time:
  //Serial.print("Unix time = ");
  // Unix time starts on Jan 1 1970. In seconds, that's 2208988800:
  const uint32_t seventyYears = 2208988800UL;
  // subtract seventy years:
  uint32_t epoch = secsSince1900 - seventyYears;
  //Serial.println(epoch);

  return epoch;
}

