/*
	Copyright (c) 2009, Noriaki Mitsunaga
    */
/* Note: this header file is coded under the policy that a programming
   beginner can easily create his/her own graphical program in C or C++.
   Then all functions and global variables are written in this file,
   which should be avoided in many other cases.
 */

#ifndef __GLIB_H__
#define __GLIB_H__
#include <windows.h>

/************************* Variables ***************************/
static HBITMAP hBitmap = NULL;
static HDC hBuffer = NULL;
static HWND ghwnd = NULL;
static HPEN dPen = NULL;
static HBRUSH dBrush = NULL;
static SIZE gws;
static int Gdelay = -1;
static CRITICAL_SECTION Gcritical_section;
static HANDLE GMainHandle;
static int GKey = -1, GMouse = -1;
static POINT GMousePos;

/************************* Functions ***************************/

inline void GWait()
{
  if (Gdelay>0)
    Sleep(Gdelay);
}

static void at_exit()
{
  /* Wait for the window to be closed */
  if (hBitmap != NULL)
    SuspendThread(GMainHandle);
}

inline int GCheckKey()
{
  int ret = -1;

  EnterCriticalSection(&Gcritical_section);
  ret = GKey;
  GKey = -1;
  LeaveCriticalSection(&Gcritical_section);

  return ret;
}

inline int GCheckLClick(int *x, int *y)
{
  int ret = -1;
  EnterCriticalSection(&Gcritical_section);
  ret = GMouse;
  GMouse = -1;
  *x = GMousePos.x;
  *y = GMousePos.y;
  LeaveCriticalSection(&Gcritical_section);

  return ret;
}

inline int GGetCursorPos(int *x, int *y)
{
  POINT pt;

  GetCursorPos(&pt);
  ScreenToClient(ghwnd, &pt);
  *x = pt.x;
  *y = pt.y;

  if (*x < 0 || *y < 0 || *x >= gws.cx || *y >= gws.cy)
    return -1;

  return 0;
}

static LRESULT CALLBACK WndProc(HWND hwnd , UINT msg , WPARAM wp , LPARAM lp)
{
	HDC hdc;
	PAINTSTRUCT ps;
	// static HBITMAP hBitmap;

	switch (msg) {
	case WM_CREATE:
		hdc = GetDC(hwnd);

		hBitmap = CreateCompatibleBitmap(hdc , gws.cx , gws.cy);
		hBuffer = CreateCompatibleDC(hdc);
		dPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
		dBrush = CreateSolidBrush(RGB(0xff, 0xff, 0xff));

		SelectObject(hBuffer , hBitmap);
		SelectObject(hBuffer , GetStockObject(NULL_PEN));
		PatBlt(hBuffer , 0 , 0 , gws.cx , gws.cy , WHITENESS);
		ReleaseDC(hwnd , hdc);
		return 0;
	case WM_CHAR:
		EnterCriticalSection(&Gcritical_section);
		GKey = (TCHAR)wp;
		LeaveCriticalSection(&Gcritical_section);
		return 0;
	case WM_DESTROY:
		DeleteDC(hBuffer);
		DeleteObject(hBitmap);
		DeleteObject(dPen);
		DeleteObject(dBrush);
		hBitmap = NULL;

		PostQuitMessage(0);
		ResumeThread(GMainHandle);
		return 0;
	case WM_LBUTTONDOWN:
		EnterCriticalSection(&Gcritical_section);
		GMouse = 0;
		GMousePos.x = LOWORD(lp);
		GMousePos.y = HIWORD(lp);
		LeaveCriticalSection(&Gcritical_section);
		return 0;
	case WM_PAINT:
		EnterCriticalSection(&Gcritical_section);
		hdc = BeginPaint(hwnd , &ps);

		BitBlt(hdc , 0 , 0 , gws.cx , gws.cy , hBuffer , 0 , 0 , SRCCOPY);

		EndPaint(hwnd , &ps);
		LeaveCriticalSection(&Gcritical_section);
		return 0;
	}
	return DefWindowProc(hwnd , msg , wp , lp);
}

static DWORD WINAPI ThreadFunc(LPVOID Param)
{
	WNDCLASS winc;
	MSG msg;
        LPSIZE ws = (SIZE *)Param;

	winc.style		= CS_HREDRAW | CS_VREDRAW;
	winc.lpfnWndProc	= WndProc;
	winc.cbClsExtra	= winc.cbWndExtra	= 0;
	winc.hInstance		= NULL /*hInstance*/;
	winc.hIcon		= LoadIcon(NULL , IDI_APPLICATION);
	winc.hCursor		= LoadCursor(NULL , IDC_ARROW);
	winc.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
	winc.lpszMenuName	= NULL;
	winc.lpszClassName	= TEXT("GLIB");

	if (!RegisterClass(&winc)) return 0;

	ghwnd = CreateWindow(
			TEXT("GLIB") , TEXT("GLIB output") ,
			(WS_OVERLAPPEDWINDOW | WS_VISIBLE) & ~WS_THICKFRAME & ~WS_MAXIMIZEBOX ,
			CW_USEDEFAULT , CW_USEDEFAULT,
			ws->cx + GetSystemMetrics(SM_CXFIXEDFRAME) * 2,
			ws->cy + GetSystemMetrics(SM_CYFIXEDFRAME) * 2 + GetSystemMetrics(SM_CYCAPTION),
			NULL , NULL ,
			NULL /* hInstance */ , NULL
	);
	/* Resume main thread */
	ResumeThread(GMainHandle);

	if (ghwnd == NULL)
		return 1;

	while(GetMessage(&msg , NULL , 0 , 0)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	return 0;
}

inline int GCreateWindow(int width, int height)
{
    DWORD threadID;

    gws.cx = width;
    gws.cy = height;

    DuplicateHandle(GetCurrentProcess(), GetCurrentThread(), GetCurrentProcess(),
		    &GMainHandle, DUPLICATE_SAME_ACCESS, FALSE, 0);
    InitializeCriticalSection(&Gcritical_section);

    CreateThread(NULL, 0, ThreadFunc, (LPVOID)&gws, 0, &threadID);
    /* Wait for ThreadFunc to finish CreateWindow() */
    SuspendThread(GMainHandle);
    atexit(at_exit);

    return 0;
}

/******************************* Drawing functions *********************/
#define GOT_ELLIPSE 0
#define GOT_RECTANGLE 1

inline void
GDrawObjectFC(int oType,
	      int x1, int y1, int x2, int y2,
	      int R, int G, int B,
	      int bR, int bG, int bB)
{
    HPEN hp;
    HBRUSH hb = NULL;

    EnterCriticalSection(&Gcritical_section);
    hp = CreatePen(PS_SOLID, 1, RGB(R, G, B));
    SelectObject(hBuffer, hp);
    if (bR>=0 && bG>=0 && bB>=0) {
      hb = CreateSolidBrush(RGB(bR, bG, bB));
      SelectObject(hBuffer, hb);
    } else {
      SelectObject(hBuffer, GetStockObject(NULL_BRUSH));
    }
    switch (oType) {
    case GOT_ELLIPSE:
      Ellipse(hBuffer, x1, y1, x2, y2);
      break;
    case GOT_RECTANGLE:
      Rectangle(hBuffer, x1, y1, x2, y2);
      break;
    }
    InvalidateRect(ghwnd , NULL , FALSE);
    DeleteObject(hp);
    if (hb != NULL)
      DeleteObject(hb);
    LeaveCriticalSection(&Gcritical_section);

    GWait();
}


/************************* Ellipse *****************/
inline void
GEllipseC(int x1, int y1, int x2, int y2, int R, int G, int B)
{
  GDrawObjectFC(GOT_ELLIPSE, x1, y1, x2, y2, R, G, B, -1, -1, -1);
}

inline void
GEllipse(int x1, int y1, int x2, int y2)
{
  GDrawObjectFC(GOT_ELLIPSE, x1, y1, x2, y2, 0, 0, 0, -1, -1, -1);
}

inline void
GEllipseFC(int x1, int y1, int x2, int y2, int R, int G, int B)
{
  GDrawObjectFC(GOT_ELLIPSE, x1, y1, x2, y2, R, G, B, R, G, B);
}

inline void
GEllipseF(int x1, int y1, int x2, int y2)
{
  GDrawObjectFC(GOT_ELLIPSE, x1, y1, x2, y2, 0, 0, 0, 0, 0, 0);
}

inline void
GEllipseFCC(int x1, int y1, int x2, int y2, int R, int G, int B, int bR, int bG, int bB)
{
  GDrawObjectFC(GOT_ELLIPSE, x1, y1, x2, y2, R, G, B, bR, bG, bB);
}

/************************* Line *****************/
inline void
GMoveTo(int x, int y)
{
  EnterCriticalSection(&Gcritical_section);
  MoveToEx(hBuffer, x, y, NULL);
  LeaveCriticalSection(&Gcritical_section);
}

inline void
GLineToC(int x, int y, int R, int G, int B)
{
  HPEN hp;

  EnterCriticalSection(&Gcritical_section);
  hp = CreatePen(PS_SOLID, 1, RGB(R, G, B));
  SelectObject(hBuffer, hp);
  LineTo(hBuffer, x, y);
  InvalidateRect(ghwnd, NULL, FALSE);
  DeleteObject(hp);
  LeaveCriticalSection(&Gcritical_section);
  GWait();
}

inline void
GLineTo(int x, int y)
{
  GLineToC(x, y, 0, 0, 0);
}

inline void
GLineC(int x1, int y1, int x2, int y2, int R, int G, int B)
{
  GMoveTo(x1, y1);
  GLineToC(x2, y2, R, G, B);
}

inline void
GLine(int x1, int y1, int x2, int y2)
{
  GLineC(x1, y1, x2, y2, 0, 0, 0);
}

/************************* Pixel *****************/
inline void
GSetPixelC(int x1, int y1, int R, int G, int B)
{
  EnterCriticalSection(&Gcritical_section);
  SetPixel(hBuffer, x1, y1, RGB(R, G, B));
  InvalidateRect(ghwnd, NULL, FALSE);
  LeaveCriticalSection(&Gcritical_section);
  GWait();
}

inline void
GSetPixel(int x1, int y1)
{
  GSetPixelC(x1, y1, 0, 0, 0);
}

/************************* Rectangle *****************/
inline void
GRectangleC(int x1, int y1, int x2, int y2, int R, int G, int B)
{
  GDrawObjectFC(GOT_RECTANGLE, x1, y1, x2, y2, R, G, B, -1, -1, -1);
}

inline void
GRectangle(int x1, int y1, int x2, int y2)
{
  GDrawObjectFC(GOT_RECTANGLE, x1, y1, x2, y2, 0, 0, 0, -1, -1, -1);
}

inline void
GRectangleFC(int x1, int y1, int x2, int y2, int R, int G, int B)
{
  GDrawObjectFC(GOT_RECTANGLE, x1, y1, x2, y2, R, G, B, R, G, B);
}

inline void
GRectangleF(int x1, int y1, int x2, int y2)
{
  GDrawObjectFC(GOT_RECTANGLE, x1, y1, x2, y2, 0, 0, 0, 0, 0, 0);
}

inline void
GRectangleFCC(int x1, int y1, int x2, int y2, int R, int G, int B, int bR, int bG, int bB)
{
  GDrawObjectFC(GOT_RECTANGLE, x1, y1, x2, y2, R, G, B, bR, bG, bB);
}

/************************* Text  *********************/
inline void
GTextC(int x, int y, const char *str, int R, int G, int B)
{
  EnterCriticalSection(&Gcritical_section);
  SetTextColor(hBuffer, RGB(R, G, B));
  TextOut(hBuffer, x, y, str, strlen(str));
  InvalidateRect(ghwnd , NULL , FALSE);
  LeaveCriticalSection(&Gcritical_section);

  GWait();
}

inline void
GText(int x, int y, const char *str)
{
  GTextC(x, y, str, 0, 0, 0);
}

inline void
GPutChar(int x, int y, const char c)
{
  char str[2] = {c, 0};
  GTextC(x, y, str, 0, 0, 0);
}

inline void
GPutCharC(int x, int y, const char c, int R, int G, int B)
{
  char str[2] = {c, 0};
  GTextC(x, y, str, R, G, B);
}

/************************* Image related functions *****************/
static void
GLoadBMPXY(const char *FileName, int x, int y)
{
	HDC hMemDC;
	HBITMAP hBitmap;
	BITMAP bitmap;


	EnterCriticalSection(&Gcritical_section);

	hMemDC = CreateCompatibleDC(NULL);
	hBitmap = (HBITMAP)LoadImage(NULL, FileName, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE);
	SelectObject(hMemDC, hBitmap);
	GetObject(hBitmap, sizeof(BITMAP), &bitmap);
	BitBlt(hBuffer, x, y, bitmap.bmWidth, bitmap.bmHeight, hMemDC, 0, 0, SRCCOPY);
	DeleteDC(hMemDC);
	DeleteObject(hBitmap);

    LeaveCriticalSection(&Gcritical_section);
    InvalidateRect(ghwnd , NULL , FALSE);
}

inline void
GLoadBMP(const char *FileName)
{
	GLoadBMPXY(FileName, 0, 0);
}

static void
GSaveBMP(const char *FileName)
{
  BITMAPFILEHEADER BMPFileHeader;
  BITMAPINFO      bi;
  DIBSECTION       DIB;
  DWORD            Work;
  HANDLE           hFile;
  HBITMAP          hbm;
  HDC hdc;
  HGDIOBJ hbmOld;
  int              Colors = 0;
  RGBQUAD          *RGBQuad;
  HDC              hMem, hDC;
  VOID *pvBits;

  // Open file
  if ((hFile = CreateFileA(FileName, GENERIC_WRITE, 0, NULL,
			  CREATE_ALWAYS,FILE_ATTRIBUTE_NORMAL |
			  FILE_FLAG_WRITE_THROUGH, NULL)) == NULL)
    return;

  // ------ Copy bitmap to DIB
  hdc = CreateCompatibleDC(hBuffer);
  ZeroMemory(&bi.bmiHeader, sizeof(BITMAPINFOHEADER));
  bi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bi.bmiHeader.biWidth        = gws.cx;
  bi.bmiHeader.biHeight       = gws.cy;
  bi.bmiHeader.biPlanes       = 1;
  bi.bmiHeader.biBitCount     = 24;
  // Create DIB section as bitmap header claims
  hbm = CreateDIBSection(hdc, &bi, DIB_RGB_COLORS, &pvBits, NULL, 0);
  // Select object
  hbmOld = SelectObject(hdc, hbm);
  // Copy from hBuffer to hdc (DIB)
  BitBlt(hdc, 0, 0, gws.cx, gws.cy, hBuffer, 0, 0, SRCCOPY);
  // get DIB
  GetObject(hbm, sizeof(DIBSECTION), &DIB);

  // Calculate size of color pallete
  if (DIB.dsBmih.biClrUsed == 0) {
    Colors = 1 << DIB.dsBmih.biClrUsed;
    switch (DIB.dsBmih.biBitCount)
      {
      case 1:  Colors=2;   break;
      case 4:  Colors=16;  break;
      case 8:  Colors=256; break;
      }
  } else {
    switch (DIB.dsBmih.biBitCount)
      {
      case 1:  Colors = DIB.dsBmih.biClrUsed;  break;
      case 4:  Colors = DIB.dsBmih.biClrUsed;  break;
      case 8:  Colors = DIB.dsBmih.biClrUsed;  break;
      }
  }
  // set BITMAPFILEHEADER
  ZeroMemory(&BMPFileHeader,sizeof(BITMAPFILEHEADER));
  BMPFileHeader.bfType = 0x4d42;
  BMPFileHeader.bfOffBits = sizeof(BITMAPFILEHEADER)
    + sizeof(BITMAPINFOHEADER) + Colors*sizeof(RGBQUAD);
  BMPFileHeader.bfSize = BMPFileHeader.bfOffBits + DIB.dsBmih.biSizeImage;

  // write BITMAPFILEHEADER
  WriteFile(hFile,&BMPFileHeader,sizeof(BITMAPFILEHEADER), &Work, NULL);
  // write BITMAPINFOHEADER
  WriteFile(hFile,&(DIB.dsBmih),sizeof(BITMAPINFOHEADER), &Work, NULL);
  // write color pallete
  if (Colors != 0) {
    hDC = GetDC(0);
    hMem = CreateCompatibleDC(hDC);
    SelectObject(hMem, hbm);
    RGBQuad = (RGBQUAD *)malloc(Colors*sizeof(RGBQUAD));
    GetDIBColorTable(hMem,0,Colors,RGBQuad);
    WriteFile(hFile,RGBQuad,Colors*sizeof(RGBQUAD), &Work, NULL);
    free(RGBQuad);
    DeleteDC(hMem);
    ReleaseDC(0, hDC);
  }
  // write image section (DIB)
  WriteFile(hFile, DIB.dsBm.bmBits, DIB.dsBmih.biSizeImage, &Work, NULL);
  // close file
  CloseHandle(hFile);

  // Deselect hbm
  SelectObject(hdc, hbmOld);
  // Delete hbm
  DeleteObject(hbm);
  // Then, delete hdc
  DeleteDC(hdc);
}

/************************* Other functions *****************/
inline void GSetDelay(int millis)
{
  Gdelay = millis;
}
#endif
