/* Library code for Gainer and clones                                        */
/* 
  Copyright (c) 2009 Noriaki Mitsunaga

  Permission is hereby granted, free of charge, to any person obtaining 
a copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation 
the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom 
the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included 
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
IN THE SOFTWARE. */

#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>

#include "gainer-lib.h"

/* Global variables */
void (*gm_button_pressed)(void) = NULL;
void (*gm_button_released)(void) = NULL;

static int configuration = 0;

/* Function declarations */
static int hex2int(int c);

/* ////////////////////////////////////////////////////////////////////////// */
/* Functions                                                                  */
/* ////////////////////////////////////////////////////////////////////////// */
int
asy_init(int fd, speed_t speed)
{
	struct termios	t;
	int modemline;

	/* ݂̐ݒǂ */
	tcgetattr(fd, &t);

	/* {[[g̃Zbg */
	cfsetspeed(&t, speed);

	/* raw [h̃Zbg(peB`FbNȂ) */
	cfmakeraw(&t);

	/* ̓[h̐ݒ */
	t.c_iflag = (IGNBRK | IGNPAR);

	/* 䃂[h̐ݒ */
	t.c_cflag = (CS8 | CREAD | CLOCAL);

	/* 䕶̐ݒ, 0.1 bo߂邩 1 M܂ŃubN */
	t.c_cc[VTIME] = 1; /* 0.1 sec */
	t.c_cc[VMIN] = 1;  /* 1 char */

	/* Set attributes */
	tcsetattr(fd, TCSANOW, &t);

	/* Flush buffers */
	tcflush(fd, TCIOFLUSH);

	return 0;
}

int
asy_open(const char *dev)
{
  int asy;

  if((asy = open(dev, O_RDWR | O_NDELAY))<0) {
    fprintf(stderr, "Colud not open serial port.\n");
    exit(2);
  }
  return asy;
}

int
gm_ain(int asy, int *a, int n)
{
  char buf[BUFSIZ];
  int i;
  unsigned long ret;

  write(asy, "I*", 2);
  usleep(5*1000);
  if (myread(asy, buf, BUFSIZ-1)<0)
    return 0;

  // fprintf(stderr, "[%s]", buf);

  if (configuration == 1 || configuration == 3 || configuration == 8) {
    if (n>4)
      n = 4;
  } else if (configuration == 2 || configuration == 4) {
    if (n>8)
      n = 8;
  } else {
    n = 0;
  }

  for (i=0; i<n; i ++)
    a[i] = hex2int(buf[1+i*2])<<4 | hex2int(buf[1+i*2+1]);

  return n;
}

void
gm_aout(int asy, int *a, int n)
{
  char s[BUFSIZ], r[BUFSIZ];
  int i;

  if (n<0)
    return;
  if (!((configuration == 1 && n == 4) || (configuration == 2 && n == 4) ||
	(configuration == 3 && n == 8) || (configuration == 4 && n == 8)))
    return;

  for (i=0; i<n; i ++)
    if (a[i]<0)
      a[i]= 0;
    else if (a[i]>0xff)
      a[i] = 0xff;

  if (n == 4) {
    snprintf(s, BUFSIZ-1, "A%02X%02X%02X%02X*", a[0], a[1], a[2], a[3]);
  } else {
    snprintf(s, BUFSIZ-1, "A%02X%02X%02X%02X%02X%02X%02X%02X*", a[0], a[1], a[2], a[3],
	     a[4], a[5], a[6], a[7]);
  }

  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      r[0] != 'A' || r[1] != '*')
    fprintf(stderr, "%s\n", r);
}

void
gm_aout1(int asy, int n, int a)
{
  char s[BUFSIZ], r[BUFSIZ];

  if (n<0)
    return;
  if (!((configuration == 1 && n<4) || (configuration == 2 && n<4) ||
	(configuration == 3 && n<8) || (configuration == 4 && n<8)))
    return;

  if (a<0)
    a = 0;
  else if (a>0xff)
    a = 0xff;

  snprintf(s, BUFSIZ-1, "a%X%02X*", n, a);
  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 || 
      strcmp(s, r) != 0)
    fprintf(stderr, "%s\n", r);
}

/*
  KONFIGURATION_n	KONFIGURATION_n*	n: configuration (1~8)

  configuration	ain/din/aout/dout	comments
  0	 0/ 0/ 0/ 0	initial configuration just after rebooting
  1	 4/ 4/ 4/ 4	default configuration
  2	 8/ 0/ 4/ 4	
  3	 4/ 4/ 8/ 0	
  4	 8/ 0/ 8/ 0	
  5	 0/16/ 0/ 0	
  6	 0/ 0/ 0/16	
  7	 0/ 8/ 8/ 0	matrix LED control mode
  [Gainer]
  8	 0/ 8/ 0/ 8	capacitive sensing switch mode (first 4 ports only)
  [GainerMini]
  8	 4/ 0/ 0/ 4	4 ain, 4 dout, 8 R/C servo
*/
int
gm_configuration(int asy, int n)
{
  char s[BUFSIZ], r[BUFSIZ];
  int rn;

  if (n<0 || n>8)
    return -1;

  snprintf(s, BUFSIZ-1, "KONFIGURATION_%d*", n);
  write(asy, s, strlen(s));
  usleep(100*1000);
  rn = myread(asy, r, BUFSIZ-1);
  if (rn<=0 || strcmp(s, r) != 0)
    return -1;

  configuration = n;
  return 0;
}

unsigned long
gm_din(int asy)
{
  char buf[BUFSIZ];
  int i;
  unsigned long ret;

  write(asy, "R*", 2);
  usleep(5*1000);
  if (myread(asy, buf, BUFSIZ-1)<0)
    return 0;
  fprintf(stderr, "[%s]", buf);
  if (buf[0] != 'R' || buf[5] != '*')
    return 0;
  if (configuration == 1 || configuration == 3)
    ret = hex2int(buf[4]);

  if (configuration == 5)
    ret = (hex2int(buf[3])<<4) | (hex2int(buf[4]));

  return ret;
}

void
gm_dH(int asy, int n)
{
  char s[BUFSIZ], r[BUFSIZ];

  if (n<0)
    return;
  if (!((configuration == 1 && n < 4) || (configuration == 2 && n < 4) ||
	(configuration == 6 && n < 0x10) || (configuration == 8 && n < 4)))
    return;

  snprintf(s, BUFSIZ-1, "H%X*", n);
  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      strcmp(s, r) != 0)
    fprintf(stderr, "%s\n", r);
}

void
gm_dL(int asy, int n)
{
  char s[BUFSIZ], r[BUFSIZ];

  if (n<0 || n>=0x10)
    return;

  snprintf(s, BUFSIZ-1, "L%X*", n);
  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      strcmp(s, r) != 0)
    fprintf(stderr, "%s\n", r);
}

void
gm_dout(int asy, int d)
{
  char s[BUFSIZ], r[BUFSIZ];

  snprintf(s, BUFSIZ-1, "D%04X*", d);
  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 || 
      strcmp(s, r) != 0)
    fprintf(stderr, "%s\n", r);
}

void
gm_led_on(int asy)
{
  char r[BUFSIZ];

  write(asy, "h*", 2);
  usleep(50*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      strcmp(r, "h*") != 0)
    fprintf(stderr, "Sent %s. Got %s.\n", "h*", r);
  return;
}

void
gm_led_off(int asy)
{
  char r[BUFSIZ];

  write(asy, "l*", 2);
  usleep(50*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      strcmp(r, "l*") != 0)
    fprintf(stderr, "Sent %s. Got %s.\n", "l*", r);
  return;
}

void
gm_mled(int asy, int n, int *row)
{
  char s[BUFSIZ], r[BUFSIZ];
  int i;

  if (n<0 || n>=8 || configuration != 7)
    return;

  for (i=0; i<n; i ++)
    if (row[i]<0)
      row[i]= 0;
    else if (row[i]>0xf)
      row[i] = 0xf;

  snprintf(s, BUFSIZ-1, "a%X%X%X%X%X%X%X%X%X*", n, 
	   row[0], row[1], row[2], row[3],
	   row[4], row[5], row[6], row[7]);

  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      r[0] != 'a' || r[1] != '*')
    fprintf(stderr, "%s\n", r);
}

void
gm_mled88(int asy, char *led)
{
  char s[BUFSIZ], r[BUFSIZ];
  int i, j;

  if (configuration != 7)
    return;

  for (i=0; i<8; i ++) {
    for (j=0; j<8; j ++)
      if (led[i*8 + j]<0)
	led[i*8 + j]= 0;
      else if (led[i*8 + j]>0xf)
	led[i*8 + j] = 0xf;

    snprintf(s, BUFSIZ-1, "a%X%X%X%X%X%X%X%X%X*", i, 
	     led[i*8 + 0], led[i*8 + 1], led[i*8 + 2], led[i*8 + 3],
	     led[i*8 + 4], led[i*8 + 5], led[i*8 + 6], led[i*8 + 7]);
    write(asy, s, strlen(s));
    usleep(5*1000); // * 𑗂AɎ̃R}h𑗂Ă͂Ȃ(?!)
  }
  usleep(5*1000);
  for (i=0; i<8; i ++) {
    if (myread(asy, r, BUFSIZ-1)<0 ||
	r[0] != 'a' || r[1] != '*')
      fprintf(stderr, "%s\n", r);
  }
}

int
gm_open(const char *dev, int mode)
{
  int asy;

  asy = asy_open(dev);
  /*
    speed: 38400bps, stop bit 1, no parity, no flow control
    all values are in hex and should be in upper case (e.g. 0A, F8...)
  */
  asy_init(asy, B38400);

  usleep(100*1000);
  gm_reset(asy);
  gm_configuration(asy, mode);

  return asy;
}

void
gm_rc(int asy, int *a, int n)
{
  char s[BUFSIZ], r[BUFSIZ];
  int i;

  if (n<0)
    return;
  if (!(configuration == 8 && n == 8))
    return;

  for (i=0; i<n; i ++)
    if (a[i]<0)
      a[i]= 0;
    else if (a[i]>0xff)
      a[i] = 0xff;

  snprintf(s, BUFSIZ-1, "P%02X%02X%02X%02X%02X%02X%02X%02X*", a[0], a[1], a[2], a[3],
	   a[4], a[5], a[6], a[7]);

  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      r[0] != 'P' || r[1] != '*')
    fprintf(stderr, "%s\n", r);
}

void
gm_rc1(int asy, int n, int a)
{
  char s[BUFSIZ], r[BUFSIZ];

  if (n<0)
    return;
  if (!(configuration == 8 && n < 8))
    return;

  if (a<0)
    a = 0;
  else if (a>0xff)
    a = 0xff;

  snprintf(s, BUFSIZ-1, "p%X%02X*", n, a);
  write(asy, s, strlen(s));
  usleep(5*1000);
  if (myread(asy, r, BUFSIZ-1)<0 ||
      strcmp(s, r) != 0)
    fprintf(stderr, "%s\n", r);
}

void
gm_reset(int asy)
{
  const char reset[] = "Q*";
  char buf[BUFSIZ];

  write(asy, reset, 2);
  write(asy, reset, 2);
  usleep(100*1000);
  tcflush(asy, TCIFLUSH); /* ignore anything recieved */
  usleep(100*1000);

  write(asy, reset, 2);
  usleep(100*1000);
  if (myread(asy, buf, BUFSIZ - 1) < 0 ||
      strcmp(buf, reset) != 0) {
    fprintf(stderr, "Reset command (%s) failed\n", reset);
    exit(1);
  }
}

int
myread(int fd, char buf[], int len)
{
  char r;
  int i;

  for (i=0; i<(len-1); i ++) {
    if (read(fd, &r, 1) <= 0) {
      // fprintf(stderr, "Read timeout\n");
      return(-1);
    }
    buf[i] = r;
    if (r == '*') {
      buf[i+1] = '\0';
      if (i == 1 && buf[0] == 'N' && buf[1] == '*') {
	if (gm_button_pressed != NULL)
	  (*gm_button_pressed)();
	i = -1;
	continue;
      } else if (i == 1 && buf[0] == 'F' && buf[1] == '*') {
	if (gm_button_released != NULL)
	  (*gm_button_released)();
	i = -1;
	continue;
      }

      return i+1;
    }
  }
  return -1;
}


int hex2int(int c)
{
  if (c>='0' && c<='9')
    return c - '0';

  return c - 'A' + 0xa;
}

/*
  Below, I put my memorandoms. 

  Programable Gain (Note: meaning is different for Gainer mini. 
                    See Gainer mini's source code)
  Gxy	Gxy*	set PGA gain and PGA reference, x: gain, y: reference

  Misc.		
  Vx	Vx*	verbose mode, x: mode (0: disabled, 1: enabled)
  ?	?1.0.0.xx*	version number, xx: build number
*/
